***************************************************************
* Simulation
***************************************************************

***************************************************************
* Read in data
***************************************************************
	
set more off
clear
* Put name of your main directory path here:
local main = "___"
cd `main'
use data_monthly.dta, clear

***************************************************************
* Set color scheme (optional)
***************************************************************

set scheme s1color

***************************************************************
* Data work
***************************************************************

* Define measure of "CORE" (Median, XFE, and so on)
* Defined measure of HEADLINE (CPI, PCE, and so on)
g CORE			= Median
g HEADLINE		= CPI
g CORE_12		= Median_12
g HEADLINE_12	= CPI_12

* NB: If using PCE measures, need to convert inflation expectations to PCE expetations as follows:
* (Comment this out if using CPI.)
*replace pe = pepce
 
* Relative price fluctuations (supply fluctuations)
g z			= HEADLINE - CORE
* 12-month average
gen z_12ma	= (1/12)*(z + z[_n-1] + z[_n-2] + z[_n-3] + z[_n-4] + z[_n-5] + z[_n-6] + z[_n-7] + z[_n-8] + z[_n-9] + z[_n-10] + z[_n-11])
* Lag
g L1z		= z[_n-1]

* Dates and sample
local latest = tm(2022m9)	
local latestpe = tm(2022m9)
local latestbc = tm(2022m8)

* Sample dummies
forval s = 1/4 {
	g sample`s' = 0
}
replace sample1 = 1 if (t>=tm(1960m1)) & (t<=tm(1972m12))
replace sample2 = 1 if (t>=tm(1973m1)) & (t<=tm(1984m12))
replace sample3 = 1 if (t>=tm(1985m1)) & (t<=tm(2019m12))
replace sample4 = 1 if (t>=tm(2020m1)) & (t<=`latest')

replace qsample = qsample[_n-4] if t>=tm(2022m12)
replace qsample = 1 if t==tm(2024m12)
replace qsample = 1 if t==tm(2025m12)

***************************************************************
* SIMULATION ASSUMPTIONS
***************************************************************

***************************************************************
***** Assumption #1. Unemployment rate path *****
***************************************************************
//* Projections as in IMF July 2022 WEO.
replace u = UWEO if t>`latest'
*/
* Interpolate ("int") between missing datapoints
ipolate u t if t>=`latest', generate(uint)
replace u = uint if t>=`latest' & u==.

***************************************************************
* Save simulation dataset with this U path and run simulations
***************************************************************

save simdata, replace

******************************************************************************************************************************
* SIMULATION
******************************************************************************************************************************

use simdata.dta, clear
local sim = 1
local name = "Pessim. BC, expectations revert"
	
***************************************************************
***** Assumption #2. Headline inflation shocks *****
***************************************************************
* Set "z" to zero:
replace z	= 0	if t>`latest'	
* 12-month average
replace z_12ma	= (1/12)*(z + z[_n-1] + z[_n-2] + z[_n-3] + z[_n-4] + z[_n-5] + z[_n-6] + z[_n-7] + z[_n-8] + z[_n-9] + z[_n-10] + z[_n-11]) if z_12ma==.

***************************************************************
***** Assumption #3. Beveridge Curve (BC) *****
***************************************************************
* Note: rho = 1 stay on COVID BC; rho = 0.75 switch gradually to pre-COVID BC
*local rho		= 0.75	
local rho		= 1

***************************************************************
***** Assumption #4. Expectations anchoring *****
***************************************************************
* Assumed feedback from inflation to change in expected inflation
* Note: feedback = 0 is fully anchored if return>0; feedback = 0 is no change in expectations if return = 0. 
* Note: feedback = 1 is backward-looking. Derive monthly gammaM by taking cubed root.
*local gammaQ	= 0.980
local gammaQ	= 0.945
local gammaM	= ((`gammaQ')^(1/3))
di(`gammaM')
*local feedback	= 1 - `gammaM'
local feedback	= 0
di(`feedback')
* Can also assume exepectations gradually converge to 2.2% (the pre-COVID anchored level, corresponding to PCE = 2.0%)
* For that, set denachoring feedback to zero (feedback = 0) and return = 0.25 (close 25% of the gap per month). 
* Or: No return (return = 0) with option to de-anchor (see above).
*local return	= 0
local return	= .25
local preCOVID	= 2.2

***************************************************************
* Variables
***************************************************************

* Vacancy rate
* V/U and 12-month average
g lu	= ln(u)
g lv	= ln(v)
label variable lv "Log vacancy rate"
label variable lu "Log unemployment rate"

* Extend lv based on BC relation and assumed u path
* Relation between lv and lu
reg lv lu if t>=tm(2020m4) & t<=`latestbc', r
* Extend lv
predict lvhat
replace lv	= lvhat if lv==.
g vhat		= exp(lvhat)

* Derive v and vu
replace v = vhat if v==.
replace vu = v/u if vu==.

* BC relation between lv and lu pre-COVID
reg lv lu if t>=tm(2009m7) & t<=tm(2020m3), r
* Fitted line
predict lvhatpre
g vhatpre		= exp(lvhatpre)
* Compute path of v based on move from COVID BC to pre-COVID BC
* Convergence parameter 
g vhatgap		= vhat - vhatpre
g _vhat	= vhatpre + ( (`rho'^(t-(`latest')))*vhatgap ) if t>=`latest'+1
replace v		= _vhat if t>=`latest'+1 
* Recompute V/U
replace vu = v/u if t>`latest'+1

* 12-month average of V/U
cap drop vu_12ma
g vu_12ma	= (1/12)*(vu + vu[_n-1] + vu[_n-2] + vu[_n-3] + vu[_n-4] + vu[_n-5] + vu[_n-6] + vu[_n-7] + vu[_n-8] + vu[_n-9] + vu[_n-10] + vu[_n-11]) 

***************************************************************
* Simulate variables
***************************************************************

* Headline inflation
g headline = HEADLINE if t<=`latest'

* Phillips curve estimation
* Define PC variables:
cap drop y x*
g y		= CORE-pe
* Measure of slack
foreach var in vu	 {
	* Define measure of slack ("x")
	cap drop x*
	g x		= `var'_12ma
	g x2	= (x)^2
	g x3	= (x)^3
	* Compute nonlinear "z" terms
	g z_12ma2 = z_12ma^2
	g z_12ma3 = z_12ma^3	
	* Estimation
	reg y x x2 x3 z_12ma z_12ma2 z_12ma3 if (sample3==1|sample4==1), r
}

* Predict inflation gap
predict yhat
* Insample predict inflation rate
g phat = yhat + pe if e(sample)

* Forecast of future headline inflation
g headlineforecast	= phat if t<=`latest'

* Compute simulated ("s") inflation ("p"), change ("d") in expected inflation, and expected inflation
g sp	= CORE			if t<=`latest' 
g spe	= pe			if t<=`latestpe'
g dspe	= pe - pe[_n-1]	if t<=`latestpe'
* This requires loop over future dates
* Change in expectations (dspe) in first simulation period: substitute in "headlineforecast" into dspe equation. 
replace dspe = `feedback'*( ((1/`gammaM')*yhat + spe[_n-1] + (1/`gammaM')*z) - spe[_n-1]) + `return'*(`preCOVID' - spe[_n-1])	if t==`latest'+1
forval h = 1/50 {
	replace spe	= spe[_n-1] + dspe			if t==`latest'+`h' & t~=.`latestpe'
	replace sp	= yhat + spe				if t==`latest'+`h'
	replace headline = sp + z				if t==`latest'+`h'
	replace headlineforecast = (1/`gammaM')*yhat + spe[_n-1] + (1/`gammaM')*z 						if t==`latest'+`h'+1
	replace dspe = `feedback'*(headlineforecast - spe[_n-1]) + `return'*(`preCOVID' - spe[_n-1])	if t==`latest'+`h'+1
}	

* Compute 12-month rates ("a" for annual)
foreach var in sp headline {
	gen index_`var'		= 100 if t==tm(1989m12)
	replace index_`var'	= index_`var'[_n-1]*(1 + (((1+(`var'/100))^(1/12))-1)) if t>tm(1989m12)
	gen `var'a			= 100*(((index_`var'/index_`var'[_n-12]))-1)	
}
	
***************************************************************
* Name and save simulation results
***************************************************************

preserve
rename u su
rename v sv
rename vu svu
rename z sz
foreach var in sp spa spe headline headlinea su sv svu sz {
    rename `var' `var'`sim'
	label variable `var'`sim' "`name'"
}
keep t sp`sim' spa`sim' spe* headline`sim' headlinea`sim' su`sim' sv* svu* sz*
save sim`sim', replace
sort t
restore

******************************************************************************************************************************
* SIMULATION
******************************************************************************************************************************

use simdata.dta, clear
local sim = 2
local name = "Pessim. BC, expectations drift ({&gamma}=0.98)"
	
***************************************************************
***** Assumption #2. Headline inflation shocks *****
***************************************************************
* Set "z" to zero:
replace z	= 0	if t>`latest'	
* 12-month average
replace z_12ma	= (1/12)*(z + z[_n-1] + z[_n-2] + z[_n-3] + z[_n-4] + z[_n-5] + z[_n-6] + z[_n-7] + z[_n-8] + z[_n-9] + z[_n-10] + z[_n-11]) if z_12ma==.

***************************************************************
***** Assumption #3. Beveridge Curve (BC) *****
***************************************************************
* Note: rho = 1 stay on COVID BC; rho = 0.75 switch gradually to pre-COVID BC
*local rho		= 0.75	
local rho		= 1

***************************************************************
***** Assumption #4. Expectations anchoring *****
***************************************************************
* Assumed feedback from inflation to change in expected inflation
* Note: feedback = 0 is fully anchored if return>0; feedback = 0 is no change in expectations if return = 0. 
* Note: feedback = 1 is backward-looking. Derive monthly gammaM by taking cubed root.
local gammaQ	= 0.980
*local gammaQ	= 0.945
local gammaM	= ((`gammaQ')^(1/3))
di(`gammaM')
local feedback	= 1 - `gammaM'
*local feedback	= 0
di(`feedback')
* Can also assume exepectations gradually converge to 2.2% (the pre-COVID anchored level, corresponding to PCE = 2.0%)
* For that, set denachoring feedback to zero (feedback = 0) and return = 0.25 (close 25% of the gap per month). 
* Or: No return (return = 0) with option to de-anchor (see above).
local return	= 0
*local return	= .25
local preCOVID	= 2.2

***************************************************************
* Variables
***************************************************************

* Vacancy rate
* V/U and 12-month average
g lu	= ln(u)
g lv	= ln(v)
label variable lv "Log vacancy rate"
label variable lu "Log unemployment rate"

* Extend lv based on BC relation and assumed u path
* Relation between lv and lu
reg lv lu if t>=tm(2020m4) & t<=`latestbc', r
* Extend lv
predict lvhat
replace lv	= lvhat if lv==.
g vhat		= exp(lvhat)

* Derive v and vu
replace v = vhat if v==.
replace vu = v/u if vu==.

* BC relation between lv and lu pre-COVID
reg lv lu if t>=tm(2009m7) & t<=tm(2020m3), r
* Fitted line
predict lvhatpre
g vhatpre		= exp(lvhatpre)
* Compute path of v based on move from COVID BC to pre-COVID BC
* Convergence parameter 
g vhatgap		= vhat - vhatpre
g _vhat	= vhatpre + ( (`rho'^(t-(`latest')))*vhatgap ) if t>=`latest'+1
replace v		= _vhat if t>=`latest'+1  
* Recompute V/U
replace vu = v/u if t>`latest'+1

* 12-month average of V/U
cap drop vu_12ma
g vu_12ma	= (1/12)*(vu + vu[_n-1] + vu[_n-2] + vu[_n-3] + vu[_n-4] + vu[_n-5] + vu[_n-6] + vu[_n-7] + vu[_n-8] + vu[_n-9] + vu[_n-10] + vu[_n-11]) 

***************************************************************
* Simulate variables
***************************************************************

* Headline inflation
g headline = HEADLINE if t<=`latest'

* Phillips curve estimation
* Define PC variables:
cap drop y x*
g y		= CORE-pe
* Measure of slack
foreach var in vu	 {
	* Define measure of slack ("x")
	cap drop x*
	g x		= `var'_12ma
	g x2	= (x)^2
	g x3	= (x)^3
	* Compute nonlinear "z" terms
	g z_12ma2 = z_12ma^2
	g z_12ma3 = z_12ma^3	
	* Estimation
	reg y x x2 x3 z_12ma z_12ma2 z_12ma3 if (sample3==1|sample4==1), r
}

* Predict inflation gap
predict yhat
* Insample predict inflation rate
g phat = yhat + pe if e(sample)

* Forecast of future headline inflation
g headlineforecast	= phat if t<=`latest'

* Compute simulated ("s") inflation ("p"), change ("d") in expected inflation, and expected inflation
g sp	= CORE			if t<=`latest' 
g spe	= pe			if t<=`latestpe'
g dspe	= pe - pe[_n-1]	if t<=`latestpe'
* This requires loop over future dates
* Change in expectations (dspe) in first simulation period: substitute in "headlineforecast" into dspe equation. 
replace dspe = `feedback'*( ((1/`gammaM')*yhat + spe[_n-1] + (1/`gammaM')*z) - spe[_n-1]) + `return'*(`preCOVID' - spe[_n-1])	if t==`latest'+1
forval h = 1/50 {
	replace spe	= spe[_n-1] + dspe			if t==`latest'+`h' & t~=.`latestpe'
	replace sp	= yhat + spe				if t==`latest'+`h'
	replace headline = sp + z				if t==`latest'+`h'
	replace headlineforecast = (1/`gammaM')*yhat + spe[_n-1] + (1/`gammaM')*z 						if t==`latest'+`h'+1
	replace dspe = `feedback'*(headlineforecast - spe[_n-1]) + `return'*(`preCOVID' - spe[_n-1])	if t==`latest'+`h'+1
}	

* Compute 12-month rates ("a" for annual)
foreach var in sp headline {
	gen index_`var'		= 100 if t==tm(1989m12)
	replace index_`var'	= index_`var'[_n-1]*(1 + (((1+(`var'/100))^(1/12))-1)) if t>tm(1989m12)
	gen `var'a			= 100*(((index_`var'/index_`var'[_n-12]))-1)	
}
	
***************************************************************
* Name and save simulation results
***************************************************************

preserve
rename u su
rename v sv
rename vu svu
rename z sz
foreach var in sp spa spe headline headlinea su sv svu sz {
    rename `var' `var'`sim'
	label variable `var'`sim' "`name'"
}
keep t sp`sim' spa`sim' spe* headline`sim' headlinea`sim' su`sim' sv* svu* sz*
save sim`sim', replace
sort t
restore

******************************************************************************************************************************
* SIMULATION
******************************************************************************************************************************

use simdata.dta, clear
local sim = 3
local name = "Pessim. BC, expectations drift ({&gamma}=0.94)"
	
***************************************************************
***** Assumption #2. Headline inflation shocks *****
***************************************************************
* Set "z" to zero:
replace z	= 0	if t>`latest'	
* 12-month average
replace z_12ma	= (1/12)*(z + z[_n-1] + z[_n-2] + z[_n-3] + z[_n-4] + z[_n-5] + z[_n-6] + z[_n-7] + z[_n-8] + z[_n-9] + z[_n-10] + z[_n-11]) if z_12ma==.

***************************************************************
***** Assumption #3. Beveridge Curve (BC) *****
***************************************************************
* Note: rho = 1 stay on COVID BC; rho = 0.75 switch gradually to pre-COVID BC
*local rho		= 0.75	
local rho		= 1

***************************************************************
***** Assumption #4. Expectations anchoring *****
***************************************************************
* Assumed feedback from inflation to change in expected inflation
* Note: feedback = 0 is fully anchored if return>0; feedback = 0 is no change in expectations if return = 0. 
* Note: feedback = 1 is backward-looking. Derive monthly gammaM by taking cubed root.
*local gammaQ	= 0.980
local gammaQ	= 0.945
local gammaM	= ((`gammaQ')^(1/3))
di(`gammaM')
local feedback	= 1 - `gammaM'
*local feedback	= 0
di(`feedback')
* Can also assume exepectations gradually converge to 2.2% (the pre-COVID anchored level, corresponding to PCE = 2.0%)
* For that, set denachoring feedback to zero (feedback = 0) and return = 0.25 (close 25% of the gap per month). 
* Or: No return (return = 0) with option to de-anchor (see above).
local return	= 0
*local return	= .25
local preCOVID	= 2.2

***************************************************************
* Variables
***************************************************************

* Vacancy rate
* V/U and 12-month average
g lu	= ln(u)
g lv	= ln(v)
label variable lv "Log vacancy rate"
label variable lu "Log unemployment rate"

* Extend lv based on BC relation and assumed u path
* Relation between lv and lu
reg lv lu if t>=tm(2020m4) & t<=`latestbc', r
* Extend lv
predict lvhat
replace lv	= lvhat if lv==.
g vhat		= exp(lvhat)

* Derive v and vu
replace v = vhat if v==.
replace vu = v/u if vu==.

* BC relation between lv and lu pre-COVID
reg lv lu if t>=tm(2009m7) & t<=tm(2020m3), r
* Fitted line
predict lvhatpre
g vhatpre		= exp(lvhatpre)
* Compute path of v based on move from COVID BC to pre-COVID BC
* Convergence parameter 
g vhatgap		= vhat - vhatpre
g _vhat	= vhatpre + ( (`rho'^(t-(`latest')))*vhatgap ) if t>=`latest'+1
replace v		= _vhat if t>=`latest'+1  
* Recompute V/U
replace vu = v/u if t>`latest'+1

* 12-month average of V/U
cap drop vu_12ma
g vu_12ma	= (1/12)*(vu + vu[_n-1] + vu[_n-2] + vu[_n-3] + vu[_n-4] + vu[_n-5] + vu[_n-6] + vu[_n-7] + vu[_n-8] + vu[_n-9] + vu[_n-10] + vu[_n-11]) 

***************************************************************
* Simulate variables
***************************************************************

* Headline inflation
g headline = HEADLINE if t<=`latest'

* Phillips curve estimation
* Define PC variables:
cap drop y x*
g y		= CORE-pe
* Measure of slack
foreach var in vu	 {
	* Define measure of slack ("x")
	cap drop x*
	g x		= `var'_12ma
	g x2	= (x)^2
	g x3	= (x)^3
	* Compute nonlinear "z" terms
	g z_12ma2 = z_12ma^2
	g z_12ma3 = z_12ma^3	
	* Estimation
	reg y x x2 x3 z_12ma z_12ma2 z_12ma3 if (sample3==1|sample4==1), r
}

* Predict inflation gap
predict yhat
* Insample predict inflation rate
g phat = yhat + pe if e(sample)

* Forecast of future headline inflation
g headlineforecast	= phat if t<=`latest'

* Compute simulated ("s") inflation ("p"), change ("d") in expected inflation, and expected inflation
g sp	= CORE			if t<=`latest' 
g spe	= pe			if t<=`latestpe'
g dspe	= pe - pe[_n-1]	if t<=`latestpe'
* This requires loop over future dates
* Change in expectations (dspe) in first simulation period: substitute in "headlineforecast" into dspe equation. 
replace dspe = `feedback'*( ((1/`gammaM')*yhat + spe[_n-1] + (1/`gammaM')*z) - spe[_n-1]) + `return'*(`preCOVID' - spe[_n-1])	if t==`latest'+1
forval h = 1/50 {
	replace spe	= spe[_n-1] + dspe			if t==`latest'+`h' & t~=.`latestpe'
	replace sp	= yhat + spe				if t==`latest'+`h'
	replace headline = sp + z				if t==`latest'+`h'
	replace headlineforecast = (1/`gammaM')*yhat + spe[_n-1] + (1/`gammaM')*z 						if t==`latest'+`h'+1
	replace dspe = `feedback'*(headlineforecast - spe[_n-1]) + `return'*(`preCOVID' - spe[_n-1])	if t==`latest'+`h'+1
}	

* Compute 12-month rates ("a" for annual)
foreach var in sp headline {
	gen index_`var'		= 100 if t==tm(1989m12)
	replace index_`var'	= index_`var'[_n-1]*(1 + (((1+(`var'/100))^(1/12))-1)) if t>tm(1989m12)
	gen `var'a			= 100*(((index_`var'/index_`var'[_n-12]))-1)	
}
	
***************************************************************
* Name and save simulation results
***************************************************************

preserve
rename u su
rename v sv
rename vu svu
rename z sz
foreach var in sp spa spe headline headlinea su sv svu sz {
    rename `var' `var'`sim'
	label variable `var'`sim' "`name'"
}
keep t sp`sim' spa`sim' spe* headline`sim' headlinea`sim' su`sim' sv* svu* sz*
save sim`sim', replace
sort t
restore

******************************************************************************************************************************
* SIMULATION
******************************************************************************************************************************

use simdata.dta, clear
local sim = 4
local name = "Optim. BC, expectations revert"
	
***************************************************************
***** Assumption #2. Headline inflation shocks *****
***************************************************************
* Set "z" to zero:
replace z	= 0	if t>`latest'	
* 12-month average
replace z_12ma	= (1/12)*(z + z[_n-1] + z[_n-2] + z[_n-3] + z[_n-4] + z[_n-5] + z[_n-6] + z[_n-7] + z[_n-8] + z[_n-9] + z[_n-10] + z[_n-11]) if z_12ma==.

***************************************************************
***** Assumption #3. Beveridge Curve (BC) *****
***************************************************************
* Note: rho = 1 stay on COVID BC; rho = 0.75 switch gradually to pre-COVID BC
local rho		= 0.75	
*local rho		= 1

***************************************************************
***** Assumption #4. Expectations anchoring *****
***************************************************************
* Assumed feedback from inflation to change in expected inflation
* Note: feedback = 0 is fully anchored if return>0; feedback = 0 is no change in expectations if return = 0. 
* Note: feedback = 1 is backward-looking. Derive monthly gammaM by taking cubed root.
*local gammaQ	= 0.980
local gammaQ	= 0.945
local gammaM	= ((`gammaQ')^(1/3))
di(`gammaM')
*local feedback	= 1 - `gammaM'
local feedback	= 0
di(`feedback')
* Can also assume exepectations gradually converge to 2.2% (the pre-COVID anchored level, corresponding to PCE = 2.0%)
* For that, set denachoring feedback to zero (feedback = 0) and return = 0.25 (close 25% of the gap per month). 
* Or: No return (return = 0) with option to de-anchor (see above).
*local return	= 0
local return	= .25
local preCOVID	= 2.2

***************************************************************
* Variables
***************************************************************

* Vacancy rate
* V/U and 12-month average
g lu	= ln(u)
g lv	= ln(v)
label variable lv "Log vacancy rate"
label variable lu "Log unemployment rate"

* Extend lv based on BC relation and assumed u path
* Relation between lv and lu
reg lv lu if t>=tm(2020m4) & t<=`latestbc', r
* Extend lv
predict lvhat
replace lv	= lvhat if lv==.
g vhat		= exp(lvhat)

* Derive v and vu
replace v = vhat if v==.
replace vu = v/u if vu==.

* BC relation between lv and lu pre-COVID
reg lv lu if t>=tm(2009m7) & t<=tm(2020m3), r
* Fitted line
predict lvhatpre
g vhatpre		= exp(lvhatpre)
* Compute path of v based on move from COVID BC to pre-COVID BC
* Convergence parameter 
g vhatgap		= vhat - vhatpre
g _vhat	= vhatpre + ( (`rho'^(t-(`latest')))*vhatgap ) if t>=`latest'+1
replace v		= _vhat if t>=`latest'+1 
* Recompute V/U
replace vu = v/u if t>`latest'+1

* 12-month average of V/U
cap drop vu_12ma
g vu_12ma	= (1/12)*(vu + vu[_n-1] + vu[_n-2] + vu[_n-3] + vu[_n-4] + vu[_n-5] + vu[_n-6] + vu[_n-7] + vu[_n-8] + vu[_n-9] + vu[_n-10] + vu[_n-11]) 

***************************************************************
* Simulate variables
***************************************************************

* Headline inflation
g headline = HEADLINE if t<=`latest'

* Phillips curve estimation
* Define PC variables:
cap drop y x*
g y		= CORE-pe
* Measure of slack
foreach var in vu	 {
	* Define measure of slack ("x")
	cap drop x*
	g x		= `var'_12ma
	g x2	= (x)^2
	g x3	= (x)^3
	* Compute nonlinear "z" terms
	g z_12ma2 = z_12ma^2
	g z_12ma3 = z_12ma^3	
	* Estimation
	reg y x x2 x3 z_12ma z_12ma2 z_12ma3 if (sample3==1|sample4==1), r
}

* Predict inflation gap
predict yhat
* Insample predict inflation rate
g phat = yhat + pe if e(sample)

* Forecast of future headline inflation
g headlineforecast	= phat if t<=`latest'

* Compute simulated ("s") inflation ("p"), change ("d") in expected inflation, and expected inflation
g sp	= CORE			if t<=`latest' 
g spe	= pe			if t<=`latestpe'
g dspe	= pe - pe[_n-1]	if t<=`latestpe'
* This requires loop over future dates
* Change in expectations (dspe) in first simulation period: substitute in "headlineforecast" into dspe equation. 
replace dspe = `feedback'*( ((1/`gammaM')*yhat + spe[_n-1] + (1/`gammaM')*z) - spe[_n-1]) + `return'*(`preCOVID' - spe[_n-1])	if t==`latest'+1
forval h = 1/50 {
	replace spe	= spe[_n-1] + dspe			if t==`latest'+`h' & t~=.`latestpe'
	replace sp	= yhat + spe				if t==`latest'+`h'
	replace headline = sp + z				if t==`latest'+`h'
	replace headlineforecast = (1/`gammaM')*yhat + spe[_n-1] + (1/`gammaM')*z 						if t==`latest'+`h'+1
	replace dspe = `feedback'*(headlineforecast - spe[_n-1]) + `return'*(`preCOVID' - spe[_n-1])	if t==`latest'+`h'+1
}	

* Compute 12-month rates ("a" for annual)
foreach var in sp headline {
	gen index_`var'		= 100 if t==tm(1989m12)
	replace index_`var'	= index_`var'[_n-1]*(1 + (((1+(`var'/100))^(1/12))-1)) if t>tm(1989m12)
	gen `var'a			= 100*(((index_`var'/index_`var'[_n-12]))-1)	
}
	
***************************************************************
* Name and save simulation results
***************************************************************

preserve
rename u su
rename v sv
rename vu svu
rename z sz
foreach var in sp spa spe headline headlinea su sv svu sz {
    rename `var' `var'`sim'
	label variable `var'`sim' "`name'"
}
keep t sp`sim' spa`sim' spe* headline`sim' headlinea`sim' su`sim' sv* svu* sz*
save sim`sim', replace
sort t
restore


******************************************************************************************************************************
* SIMULATION
******************************************************************************************************************************

use simdata.dta, clear
local sim = 5
local name = "Optim. BC, expectations drift ({&gamma}=0.98)"
	
***************************************************************
***** Assumption #2. Headline inflation shocks *****
***************************************************************
* Set "z" to zero:
replace z	= 0	if t>`latest'	
* 12-month average
replace z_12ma	= (1/12)*(z + z[_n-1] + z[_n-2] + z[_n-3] + z[_n-4] + z[_n-5] + z[_n-6] + z[_n-7] + z[_n-8] + z[_n-9] + z[_n-10] + z[_n-11]) if z_12ma==.

***************************************************************
***** Assumption #3. Beveridge Curve (BC) *****
***************************************************************
* Note: rho = 1 stay on COVID BC; rho = 0.75 switch gradually to pre-COVID BC
local rho		= 0.75	
*local rho		= 1

***************************************************************
***** Assumption #4. Expectations anchoring *****
***************************************************************
* Assumed feedback from inflation to change in expected inflation
* Note: feedback = 0 is fully anchored if return>0; feedback = 0 is no change in expectations if return = 0. 
* Note: feedback = 1 is backward-looking. Derive monthly gammaM by taking cubed root.
local gammaQ	= 0.980
*local gammaQ	= 0.945
local gammaM	= ((`gammaQ')^(1/3))
di(`gammaM')
local feedback	= 1 - `gammaM'
*local feedback	= 0
di(`feedback')
* Can also assume exepectations gradually converge to 2.2% (the pre-COVID anchored level, corresponding to PCE = 2.0%)
* For that, set denachoring feedback to zero (feedback = 0) and return = 0.25 (close 25% of the gap per month). 
* Or: No return (return = 0) with option to de-anchor (see above).
local return	= 0
*local return	= .25
local preCOVID	= 2.2

***************************************************************
* Variables
***************************************************************

* Vacancy rate
* V/U and 12-month average
g lu	= ln(u)
g lv	= ln(v)
label variable lv "Log vacancy rate"
label variable lu "Log unemployment rate"

* Extend lv based on BC relation and assumed u path
* Relation between lv and lu
reg lv lu if t>=tm(2020m4) & t<=`latestbc', r
* Extend lv
predict lvhat
replace lv	= lvhat if lv==.
g vhat		= exp(lvhat)

* Derive v and vu
replace v = vhat if v==.
replace vu = v/u if vu==.

* BC relation between lv and lu pre-COVID
reg lv lu if t>=tm(2009m7) & t<=tm(2020m3), r
* Fitted line
predict lvhatpre
g vhatpre		= exp(lvhatpre)
* Compute path of v based on move from COVID BC to pre-COVID BC
* Convergence parameter 
g vhatgap		= vhat - vhatpre
g _vhat	= vhatpre + ( (`rho'^(t-(`latest')))*vhatgap ) if t>=`latest'+1
replace v		= _vhat if t>=`latest'+1 
* Recompute V/U
replace vu = v/u if t>`latest'+1

* 12-month average of V/U
cap drop vu_12ma
g vu_12ma	= (1/12)*(vu + vu[_n-1] + vu[_n-2] + vu[_n-3] + vu[_n-4] + vu[_n-5] + vu[_n-6] + vu[_n-7] + vu[_n-8] + vu[_n-9] + vu[_n-10] + vu[_n-11]) 

***************************************************************
* Simulate variables
***************************************************************

* Headline inflation
g headline = HEADLINE if t<=`latest'

* Phillips curve estimation
* Define PC variables:
cap drop y x*
g y		= CORE-pe
* Measure of slack
foreach var in vu	 {
	* Define measure of slack ("x")
	cap drop x*
	g x		= `var'_12ma
	g x2	= (x)^2
	g x3	= (x)^3
	* Compute nonlinear "z" terms
	g z_12ma2 = z_12ma^2
	g z_12ma3 = z_12ma^3	
	* Estimation
	reg y x x2 x3 z_12ma z_12ma2 z_12ma3 if (sample3==1|sample4==1), r
}

* Predict inflation gap
predict yhat
* Insample predict inflation rate
g phat = yhat + pe if e(sample)

* Forecast of future headline inflation
g headlineforecast	= phat if t<=`latest'

* Compute simulated ("s") inflation ("p"), change ("d") in expected inflation, and expected inflation
g sp	= CORE			if t<=`latest' 
g spe	= pe			if t<=`latestpe'
g dspe	= pe - pe[_n-1]	if t<=`latestpe'
* This requires loop over future dates
* Change in expectations (dspe) in first simulation period: substitute in "headlineforecast" into dspe equation. 
replace dspe = `feedback'*( ((1/`gammaM')*yhat + spe[_n-1] + (1/`gammaM')*z) - spe[_n-1]) + `return'*(`preCOVID' - spe[_n-1])	if t==`latest'+1
forval h = 1/50 {
	replace spe	= spe[_n-1] + dspe			if t==`latest'+`h' & t~=.`latestpe'
	replace sp	= yhat + spe				if t==`latest'+`h'
	replace headline = sp + z				if t==`latest'+`h'
	replace headlineforecast = (1/`gammaM')*yhat + spe[_n-1] + (1/`gammaM')*z 						if t==`latest'+`h'+1
	replace dspe = `feedback'*(headlineforecast - spe[_n-1]) + `return'*(`preCOVID' - spe[_n-1])	if t==`latest'+`h'+1
}	

* Compute 12-month rates ("a" for annual)
foreach var in sp headline {
	gen index_`var'		= 100 if t==tm(1989m12)
	replace index_`var'	= index_`var'[_n-1]*(1 + (((1+(`var'/100))^(1/12))-1)) if t>tm(1989m12)
	gen `var'a			= 100*(((index_`var'/index_`var'[_n-12]))-1)	
}
	
***************************************************************
* Name and save simulation results
***************************************************************

preserve
rename u su
rename v sv
rename vu svu
rename z sz
foreach var in sp spa spe headline headlinea su sv svu sz {
    rename `var' `var'`sim'
	label variable `var'`sim' "`name'"
}
keep t sp`sim' spa`sim' spe* headline`sim' headlinea`sim' su`sim' sv* svu* sz*
save sim`sim', replace
sort t
restore

******************************************************************************************************************************
* SIMULATION
******************************************************************************************************************************

use simdata.dta, clear
local sim = 6
local name = "Optim. BC, expectations drift ({&gamma}=0.94)"
	
***************************************************************
***** Assumption #2. Headline inflation shocks *****
***************************************************************
* Set "z" to zero:
replace z	= 0	if t>`latest'	
* 12-month average
replace z_12ma	= (1/12)*(z + z[_n-1] + z[_n-2] + z[_n-3] + z[_n-4] + z[_n-5] + z[_n-6] + z[_n-7] + z[_n-8] + z[_n-9] + z[_n-10] + z[_n-11]) if z_12ma==.

***************************************************************
***** Assumption #3. Beveridge Curve (BC) *****
***************************************************************
* Note: rho = 1 stay on COVID BC; rho = 0.75 switch gradually to pre-COVID BC
local rho		= 0.75	
*local rho		= 1

***************************************************************
***** Assumption #4. Expectations anchoring *****
***************************************************************
* Assumed feedback from inflation to change in expected inflation
* Note: feedback = 0 is fully anchored if return>0; feedback = 0 is no change in expectations if return = 0. 
* Note: feedback = 1 is backward-looking. Derive monthly gammaM by taking cubed root.
*local gammaQ	= 0.980
local gammaQ	= 0.945
local gammaM	= ((`gammaQ')^(1/3))
di(`gammaM')
local feedback	= 1 - `gammaM'
*local feedback	= 0
di(`feedback')
* Can also assume exepectations gradually converge to 2.2% (the pre-COVID anchored level, corresponding to PCE = 2.0%)
* For that, set denachoring feedback to zero (feedback = 0) and return = 0.25 (close 25% of the gap per month). 
* Or: No return (return = 0) with option to de-anchor (see above).
local return	= 0
*local return	= .25
local preCOVID	= 2.2

***************************************************************
* Variables
***************************************************************

* Vacancy rate
* V/U and 12-month average
g lu	= ln(u)
g lv	= ln(v)
label variable lv "Log vacancy rate"
label variable lu "Log unemployment rate"

* Extend lv based on BC relation and assumed u path
* Relation between lv and lu
reg lv lu if t>=tm(2020m4) & t<=`latestbc', r
* Extend lv
predict lvhat
replace lv	= lvhat if lv==.
g vhat		= exp(lvhat)

* Derive v and vu
replace v = vhat if v==.
replace vu = v/u if vu==.

* BC relation between lv and lu pre-COVID
reg lv lu if t>=tm(2009m7) & t<=tm(2020m3), r
* Fitted line
predict lvhatpre
g vhatpre		= exp(lvhatpre)
* Compute path of v based on move from COVID BC to pre-COVID BC
* Convergence parameter 
g vhatgap		= vhat - vhatpre
g _vhat	= vhatpre + ( (`rho'^(t-(`latest')))*vhatgap ) if t>=`latest'+1
replace v		= _vhat if t>=`latest'+1 
* Recompute V/U
replace vu = v/u if t>`latest'+1

* 12-month average of V/U
cap drop vu_12ma
g vu_12ma	= (1/12)*(vu + vu[_n-1] + vu[_n-2] + vu[_n-3] + vu[_n-4] + vu[_n-5] + vu[_n-6] + vu[_n-7] + vu[_n-8] + vu[_n-9] + vu[_n-10] + vu[_n-11]) 

***************************************************************
* Simulate variables
***************************************************************

* Headline inflation
g headline = HEADLINE if t<=`latest'

* Phillips curve estimation
* Define PC variables:
cap drop y x*
g y		= CORE-pe
* Measure of slack
foreach var in vu	 {
	* Define measure of slack ("x")
	cap drop x*
	g x		= `var'_12ma
	g x2	= (x)^2
	g x3	= (x)^3
	* Compute nonlinear "z" terms
	g z_12ma2 = z_12ma^2
	g z_12ma3 = z_12ma^3	
	* Estimation
	reg y x x2 x3 z_12ma z_12ma2 z_12ma3 if (sample3==1|sample4==1), r
}

* Predict inflation gap
predict yhat
* Insample predict inflation rate
g phat = yhat + pe if e(sample)

* Forecast of future headline inflation
g headlineforecast	= phat if t<=`latest'

* Compute simulated ("s") inflation ("p"), change ("d") in expected inflation, and expected inflation
g sp	= CORE			if t<=`latest' 
g spe	= pe			if t<=`latestpe'
g dspe	= pe - pe[_n-1]	if t<=`latestpe'
* This requires loop over future dates
* Change in expectations (dspe) in first simulation period: substitute in "headlineforecast" into dspe equation. 
replace dspe = `feedback'*( ((1/`gammaM')*yhat + spe[_n-1] + (1/`gammaM')*z) - spe[_n-1]) + `return'*(`preCOVID' - spe[_n-1])	if t==`latest'+1
forval h = 1/50 {
	replace spe	= spe[_n-1] + dspe			if t==`latest'+`h' & t~=.`latestpe'
	replace sp	= yhat + spe				if t==`latest'+`h'
	replace headline = sp + z				if t==`latest'+`h'
	replace headlineforecast = (1/`gammaM')*yhat + spe[_n-1] + (1/`gammaM')*z 						if t==`latest'+`h'+1
	replace dspe = `feedback'*(headlineforecast - spe[_n-1]) + `return'*(`preCOVID' - spe[_n-1])	if t==`latest'+`h'+1
}	

* Compute 12-month rates ("a" for annual)
foreach var in sp headline {
	gen index_`var'		= 100 if t==tm(1989m12)
	replace index_`var'	= index_`var'[_n-1]*(1 + (((1+(`var'/100))^(1/12))-1)) if t>tm(1989m12)
	gen `var'a			= 100*(((index_`var'/index_`var'[_n-12]))-1)	
}
	
***************************************************************
* Name and save simulation results
***************************************************************

preserve
rename u su
rename v sv
rename vu svu
rename z sz
foreach var in sp spa spe headline headlinea su sv svu sz {
    rename `var' `var'`sim'
	label variable `var'`sim' "`name'"
}
keep t sp`sim' spa`sim' spe* headline`sim' headlinea`sim' su`sim' sv* svu* sz*
save sim`sim', replace
sort t
restore

******************************************************************************************************************************
* MERGE SIMULATIONS AND PLOT THEM
******************************************************************************************************************************

use simdata.dta, clear

***************************************************************
//* Plot results: Different simulations: CORE
***************************************************************

* Merge in the different results
forval s = 1/6 {
	sort t
	merge t using sim`s'.dta
	cap drop _merge
	replace sp`s' = . if t<`latest'
	replace headline`s' = . if t<`latest'	
	replace spa`s' = . if t<`latest'
	replace headlinea`s' = . if t<`latest'		
}

g plotsample = 1 if t>=tm(2020m1) & t<=tm(2024m12)

* Make the plots

* Pick which simulations to show in plots

local sim1 = 1
local sim2 = 2
local sim3 = 3
local sim4 = 4
local sim5 = 5
local sim6 = 6

local title = "A. Conditional on IMF Staff Unemployment Path"
*local title = ""
local index = 2
preserve
keep if plotsample==1
sum t
local t0 = r(min)
local t1 = `t0'+(5*12)
format spa* %9.2f
twoway (line CORE_12 spa`sim1' spa`sim2' spa`sim3' spa`sim4' spa`sim5' spa`sim6' t, ///
yline(2.6, lcolor(ebg) lwidth(medthick) lpattern(dash)) xline(`latest', lcolor(ebg) lwidth(medthick)) ///
lcolor(blue red red red midblue midblue midblue) ///	
lpattern(solid solid dash shortdash solid dash shortdash) ///
lwidth(medthick medthick medthick medthick medthick medthick medthick medthick medthick medthick medthick)) ///
(scatter spa`sim3' t if t==tm(2024m12), mlabposition(7) msize(small) mcolor(black) mfcolor(red) mlabel(spa`sim3') mlabcolor(black) mlabsize(small)) ///
(scatter spa`sim4' t if t==tm(2024m12), mlabposition(7) msize(small) mcolor(black) mfcolor(midblue) mlabel(spa`sim4') mlabcolor(black) mlabsize(small) ///
title("`title'", size(medium)) ///
xmtick(#19) xlabel(`t0'(12)`t1', format(%tmCY) angle(90)) xlabel(,labsize(medium)) ylabel(1(1)11,labsize(medium) angle(0)) xtitle("") ytitle("") ///
legend(row(11) symxsize(6) ring(0) position(11) bmargin(none) region(style(none))) ///
legend(order(2 3 4 5 6 7)) ///
legend(size(small)) ysize(5) ///
xsize(6) saving(`index'.gph, replace))
restore

clist t spa1 spa2 spa3 spa4 spa5 spa6 if t>=tm(2022m6) & t<=tm(2024m12), noobs	
d spa1 spa2 spa3 spa4 spa5 spa6
*/

* Save outsheet for use elsewhere
preserve
local name = "IMF"
keep if t>=tm(2022m1) & t<=tm(2024m12)
keep t spa1 spa2 spa3 spa4 spa5 spa6
foreach var in spa1 spa2 spa3 spa4 spa5 spa6 {
	rename `var' `name'_`var'
}
sort t
save `name'.dta, replace
restore

***************************************************************
* Save results 
***************************************************************

preserve
keep t CORE_12 spa1 spa2 spa3 spa4 spa5 spa6 
format CORE_12 spa1 spa2 spa3 spa4 spa5 spa6 %9.3f
keep if t>=tm(2022m1) & t<=tm(2024m12)	
outsheet using "results\charts\Figure 19 Panel A.xls", replace
restore

